/**
 * Klasse zum Verwalten der Welt (inkl. Lebewesen und Objekte).
 */
class World {
  /** Beinhaltet die Topologie der Welt */
  private Map map;
  /** Verwaltet die Partikel (Anzahl, Erzeugen usw.) */
  private ParticleManager particles;
  /** Beinhaltet alle Lebewesen */
  private ArrayList<Creature> creatures;
  /** Interne Liste von neu hinzugefügten Lebewesen, um Index-Fehler zu vermeiden (Warteschleife) */
  private ArrayList<Creature> added_creatures;
  /** Interne Liste von entfernten Lebewesen, um Index-Fehler zu vermeiden (Warteschleife) */
  private ArrayList<Creature> removed_creatures;
  /** Mittelpunkt für die Strömung */
  private PVector currentPos;
  /** Durchmesser für Breite und Höhe der Strömung */
  private PVector currentSize;
  /** PApplet der Processing Instanz */
  private PApplet applet;

  /**
   * Erzeugt die Welt, Strömung und Partikel
   *
   * @param applet PApplet der Processing Umgebung
   */
  public World(PApplet applet) {
    this.applet = applet;
    
    map = new Map(width, height);
    particles = new ParticleManager();

    // Strömungsellipse mit der Größe der Welt erzeugen
    currentPos = new PVector(this.getWidth() / 2, this.getHeight() / 2);
    currentSize = new PVector(width - 200, height - 200);

    creatures = new ArrayList<Creature>();
    this.added_creatures = new ArrayList<Creature>();
    this.removed_creatures = new ArrayList<Creature>();
  }

  /**
   * Liefert die Stärke der Ströumg an übergebenen Punkt
   *
   * @param pos Position, an der die Ströumung abgefragt werden soll
   * @return Gibt den Weg in Pixeln zurück, welcher durch die Strömung zurückgelegt wird
   */
  public Current getCurrent(PVector pos) {
    // Mittelpunkt der Ellipse
    PVector center = new PVector(currentPos.x, currentPos.y);

    // Ermittel Winkel in dem das Objekt steht
    double direction = Math.atan2(pos.y-center.y, pos.x-center.x);

    // Ermittel Punkt auf Ellipse
    PVector ellipsePoint = new PVector();
    ellipsePoint.x = center.x + (int) ((currentSize.x/2) * Math.cos(direction));
    ellipsePoint.y = center.y + (int) ((currentSize.y/2) * Math.sin(direction));

    // Ermittel von Punkt auf Ellipse Abstand zum Objekt
    double distance = pos.dist(ellipsePoint);

    // Ermittel Stärke
    double intensity = 0;

    if (distance < 80) {
      intensity = 1-distance/80;
    }

    // Ermittel die Richtung
    Current current = new Current();
    current.direction = direction - HALF_PI;
    current.intensity = intensity;

    return current;
  }

  /**
   * Gibt die Liste aller (auch toter!) in der Welt befindlichen Lebewesen zurück
   *
   * @return Liste der Lebewesen
   */
  public ArrayList<Creature> getCreatures() {
    return creatures;
  }

  /**
   * Fügt Lebewesen zur Welt hinzu.
   * Lebewesen erweitern die Klasse {@link Creature}
   * Die hinzugefügten Lebewesen befinden sich zunächst in einer Warteschleife und werden erst nach einem Tick hinzugefügt.
   *
   * @param obj Das hinzuzufügende Lebewesen
   */
  public void addCreature(Creature obj)
  { 
    this.added_creatures.add(obj);
  } 

  /**
   * Behandelt die Logik für einen Tick
   */
  public void update() {
    // Kreaturen werden aktualisiert, wenn sie nicht tot sind
    for (Creature creature : creatures) {
      if (!creature.isDead()) {
        creature.update();
        creature.physics();
        creature.updateVital();
      }
            
      else
      {
        removed_creatures.add(creature);
      }
    }

    particles.update();
    
    // Lebewesen aus der Warteschleife hinzufügen
    for (Creature c : added_creatures)
      creatures.add(c);
    
    // Lebewesen aus der Warteschleife löschen
    for (Creature c : removed_creatures)
      creatures.remove(c);
    
    // Warteschleife leeren
    added_creatures.clear();
    removed_creatures.clear();
  }

  /**
   * Methode zum Rendern der Welt und ihrer Bewohner
   */
  public void render() {
    background(255);
    ambientLight(128, 128, 128);
    directionalLight(126, 126, 126, 0, 0, -1);
    ortho();
    this.map.render();

    for (Creature creature : creatures) {
      if (!creature.isDead()) {
        creature.render();
      }
    }

    this.particles.render();
  }
  
  /**
   * Gibt die Breite der Welt zurück
   *
   * @return Breite der Welt in Pixel
   */
  public int getWidth() {
    return this.map.getWidth();
  } 

  /**
   * Gibt die Höhe der Welt zurück
   *
   * @return Höhe der Welt in Pixel
   */
  public int getHeight()
  {
    return this.map.getHeight();
  }

  /**
   * Gibt PApplet der Processing Umgebung zurück
   *
   * @return PApplet der Processing Umgebung
   */
  public PApplet getApplet()
  {
    return this.applet;
  }

  /**
   * Gibt den Partikelmanager zum Verwalten der Partikel in der Welt zurück
   *
   * @return Gibt PartikelManager zurück
   */
  public ParticleManager getParticleManager() 
  {
    return particles;
  }
}

