/**
 * Klasse zum Erstellen von Wänden mit prozedualer Topologie
 *
 * @author Martin M.
 */
class Wall
{
    /** Position des linken, oberen Eckpunktes der Wand */
    private PVector position;
    /** Breite der Wand */
    private float width;
    /** Höhe der Wand */
    private float height;
    /** Tiefe der Wand (des Reliefs) */
    private float depth;
    /** Anzahl der Unterteilungen in der Breite */
    private int uNodes;
    /** Anzahl der Unterteilungen in der Höhe */
    private int vNodes;
    /** Speichern der Vertices (X-Koordinate: Index der 1. Dimension, Y-Koordinate: Index der 2. Dimension, Z-Koordinate: Wert im Array) */
    private float[][] vertices;
    /** Rotation um die X-Achse */
    private float rotationX;
    /** Rotation um die Z-Achse */
    private float rotationZ;
    
    /**
     * Konstruktor zum Erstellen einer Wand
     *
     * @param x X-Koordinate der linken, oberen Ecke
     * @param y Y-Koordinate der linken, oberen Ecke
     * @param w Breite der Wand
     * @param h Höhe der Wand
     * @param d Tiefe der Wand (des Reliefs)
     */
    public Wall(float x, float y, float w, float h, float d)
    {
        this.position = new PVector(x, y);
        this.width = w;
        this.height = h;
        this.depth = d;
        this.uNodes = 10;
        this.vNodes = 10;
        this.vertices = new float[uNodes][vNodes];
        this.rotationX = 0;
        this.rotationZ = 0;
        
        createVertices();
    }
    
    /**
     * Konstruktor zum Erstellen einer Wand
     *
     * @param x X-Koordinate der linken, oberen Ecke
     * @param y Y-Koordinate der linken, oberen Ecke
     * @param w Breite der Wand
     * @param h Höhe der Wand
     * @param d Tiefe der Wand (des Reliefs)
     * @param rotZ Rotation um die Z-Achse
     */
    public Wall(float x, float y, float w, float h, float d, float rotZ)
    {
        this(x, y, w, h, d);
        
        this.setRotationZ(rotZ);
    }
    
    /**
     * Methode zum Erstellen der Vertices
     * Für jedes Vertice wird der Perlin-Noise an dessen Position berechnet und in dem zwei-dimensionalen Array gespeichert
     */
    public void createVertices()
    {
        float noiseScale = 0.5;
        float noiseAmplitude = 2;
        int noiseSeed = 0;
        int noiseOctaves = 4;
        float noiseFalloff = 0.5;
        float z;
        
        noiseSeed(noiseSeed);
        noiseDetail(noiseOctaves, noiseFalloff);
            
        for(int u = 0; u < this.uNodes; u++)
        {
            for(int v = 0; v < this.vNodes; v++)
            {
                if(v == 0)
                    z = 0;
                else
                    z = log(v + 1) / 2 * noise(u * noiseScale, v * noiseScale) * noiseAmplitude;
                    
                this.vertices[u][v] = z;
            }
        }
    }
    
    /**
     * Setzt die Rotation um die X-Achse
     *
     * @param angle Winkel als Radiant
     */
    public void setRotationX(float angle)
    {
        this.rotationX = angle;
    }
    
    /**
     * Setzt die Rotation um die Z-Achse
     *
     * @param angle Winkel als Radiant
     */
    public void setRotationZ(float angle)
    {
        this.rotationZ = angle;
    }
    
    /**
     * Methode zum Rendern der Wand
     */
    public void render()
    {
        noStroke();
        fill(128);
        
        pushMatrix();
        translate(this.position.x, this.position.y);
        rotateZ(this.rotationZ);
        scale(this.width / (this.uNodes - 1), this.height / (this.vNodes - 1), this.depth);
        rotateX(radians(-90));
        
        for(int v = 0; v < vNodes - 1; v++)
        {
            beginShape(TRIANGLE_STRIP);
            
            for(int u = 0; u < uNodes; u++)
            {
                vertex(u, v, this.vertices[u][v]);
                
                vertex(u, v + 1, this.vertices[u][v + 1]);
            }
            
            endShape();
        }
        
        popMatrix();
    }
}
