/** Fortlaufende ID für jedes erzeugte Lebewesen **/
private static int idCounter = 0;   

/**
 * Elternklasse für sämtliche Kreaturen in der Welt
 */
abstract class Creature {
    private float speed = 0;
    /** ID des Objektes */
    private int id;
    /** Aktuelle Position */
    private PVector pos;
    /** Sichtrichtung */
    private float facingAngle;
    /** Ist es Tot? */
    private boolean isDead = false;

    /** Geburtszeit (Nanosekunden) erzeugt durch {@link System#nanoTime()} */
    private long startTime;

    /** Lebenspunkte für jedes Level */
    private final float[] MAX_HEALTH = { 
        100, 125, 150
    };
    /** Die aktuellen Lebenspunkte */
    private float health;
    /** Speichert Welt für eventuelle Abfragen der Größe, anderer Lebewesen etc. */
    private World world;

    /** Letzte gespeicherte Zeit, um Updates in einem Intervall (z.B jede Sekunde) durchführen zu können */
    private int lastSecond;

    /**
     * Erzeugt die Kreatur.
     *
     * @param world Die Welt in der das Objekt sich befindet.
     * @param pos   Position wo Objekt entsteht.
     */
    public Creature(World world, PVector pos) 
    {
        this.id = idCounter++;

        this.world = world;
        this.pos = pos;

        this.facingAngle = 0.0;

        this.startTime = System.nanoTime();

        this.health = getMaxHealth();

        this.lastSecond = second();
    }

    /**
     * Methode zum Zurückgeben der ID des Lebewesens
     *
     * @return Einmalige ID
     */
    public final int getID() 
    {
        return id;
    }

    /**
     * Gibt Position des Lebewesens zurück
     *
     * @return Vector mit Position des Lebewesens
     */
    public final PVector getPosition() 
    {
        return pos;
    }

    /**
     * Methode zum Setzen einer beliebigen Position für das Lebewesen
     *
     * @param x X-Koordinate
     * @param y Y-Koordinate
     * @param z Z-Koordinate
     */
    public void setPosition(float x, float y, float z) 
    {
        this.pos.x = x;
        this.pos.y = y;
        this.pos.z = z;
    }

    /**
     * Gibt Blickrichtung als Winkel zurück
     *
     * @return Winkel der Blickrichtung als Radiant
     */
    public final float getFace() 
    {
        return this.facingAngle;
    }

    /**
     * Dreht Lebewesen zu übergebenen Winkel (Blickrichtung)
     *
     * @param angle Winkel als Radiant
     */
    public final void setFace(float angle) 
    {
        this.facingAngle = angle;
    }

    /**
     * Gibt die Strömung für die aktuelle Position wider.
     *
     * @return Positionsänderung durch die Strömung in X,Y in {@link Current} verpackt.
     * @see Current
     */
    public final Current getCurrent() {
        return world.getCurrent(pos);
    }

    /**
     * Wenn diese aufgerufen wird, bewegt sich das Objekt in diesem Tick.
     *
     * Bewegung durch die Strömung wird in der {@link Creature#physics()} gemacht und nach der Update automatisch aufgerufen.
     */
    public final void move() 
    {
        // Breche ab, wenn Tier schon tot ist.
        if (isDead()) {
            return;
        }

        pos.x += Math.cos(facingAngle) * getSpeed();
        pos.y += Math.sin(facingAngle) * getSpeed();
    }

    /**
     * Berechnet physikalische Einflüsse (aktuell nur Strömung)
     */
    public final void physics() {
        // Strömung
        // Strömung abholen
        Current current = world.getCurrent(pos);

        pos.x += Math.cos(current.direction) * current.intensity;
        pos.y += Math.sin(current.direction) * current.intensity;
    }

    /**
     * Berechnet vitale Zustände (Hunger und Alter)
     */
    public final void updateVital() {
        // Ist er Tot?
        if (isDead()) {
            return;
        }

        // Ist maximales Lebensalter vorbei?
        if ((System.nanoTime() - startTime) / 1000000000.0 > getMaxLifetime()) {
            setDead();
        }

        // Ist er Verhungert / durch angriffe gestorben
        if (getHealth() <= 0) {
            setDead();
        }

        // Inhalt wird nur jede Sekunde ausgeführt.
        if (lastSecond != second()) {
            lastSecond = second();

            // Hungern
            setHealth(getHealth()-(getMaxHealth()*(getMetabolism()/100)));
        }

        if (!this.isDead()) {
            if (random(0.0, 4.0) < getSpeed()/10) 
                world.getParticleManager().addParticle(this.getPosition());
        }
    }

    /**
     * Gibt alle Kreaturen im Sichtfeld aus.
     * 
     * @return Kreaturen im Sichtfeld.
     * @see Creature
     */
    public final ArrayList<Creature> inSightRadius() {
        ArrayList<Creature> inSightRadius = new ArrayList<Creature>();

        for (Creature creature: world.getCreatures()) 
        {
            // Überspringe sich selbst.
            if (creature == this) {
                continue;
            }

            if (getDistance(creature) < getSightRadius()*20) {
                inSightRadius.add(creature);
            }
        }

        return inSightRadius;
    }

    /**
     * Gibt die Distanz von diesem Objekt zum übergebenem zurück.
     *
     * @param obj Objekt, zu dem die Distanz ermittelt werden soll
     * @return Distanz zum übergenen Objekt
     */
    public double getDistance(Creature obj) {
        // Ermittel Abstand zwischen beiden Objekten.
        return pos.dist(obj.getPosition());
    }

    /**
     * Gibt die Richtung zu übergenen Objekt als Winkel zurück.
     *
     * @param obj Objekt, zu dem der Winkel berechnet werden soll
     * @return Winkel zu dem Objekt als Radiant
     */
    public double getRadiant(Creature obj) {
        PVector objPos = obj.getPosition();        
        // Ermittel Radiant in dem das Objekt steht
        return Math.atan2(pos.y-objPos.y, pos.x-objPos.x);
    }

    /**
     * Methode zum Erhalten der aktuellen Geschwindigkeit einer Kreatur
     *
     * @return Geschwindikeit
     */
    abstract public float getSpeed();
    
    /**
     * Methode zum Ermitteln der Größe des Sichtradius
     *
     * @return Radius der Sichtweite
     */
    abstract public float getSightRadius();

    /**
     * Methode zum Zurückgeben der maximalen Lebenszeit
     *
     * @return Maximale Lebenszeit in Sekunden
     */
    abstract public float getMaxLifetime();
    
    /**
     * Methode zum Ermitteln der Stoffwechselaktivität (Hunger)
     *
     * @return Rate in Prozent, um die die Gesundheit pro Sekunde verringert wird
     */
    abstract public float getMetabolism();

    /**
     * Methode zum Identifizieren vom übergeben Objekt als Feind
     *
     * @param obj Objekt zum Überprüfen
     * @return Ob das übergebene Objekt ein Feind ist
     */
    abstract public boolean isEnemy(Creature obj);
    
    /**
     * Methode zum Identifizieren von Objekten als Futter
     *
     * @param obj Objekt zum Überprüfen
     * @return Ob das übergebene Objekt fressbar ist
     */
    abstract public boolean isFood(Creature obj);

    /**
     * Methode zum Ermitteln der maximalen Gesundheit
     *
     * @return Ob das übergebene Objekt fressbar ist
     */
    public final float getHealth() {
        return health;
    }

    /**
     * Methode zum Ändern der Gesundheit
     *
     * @param health Neuer Wert für die Gesundheit
     */
    public final void setHealth(float health) {
        this.health = health;
        if (health > getMaxHealth())
            this.health = getMaxHealth();
    }

    /**
     * Methode zum Abfragen der maximalen Gesundheit
     *
     * @return Maximaler Wert der Gesundheit für das aktuelle Level
     */
    public final float getMaxHealth() {
        return MAX_HEALTH[getLevel()];
    }

    /**
     * Methode zum Abfragen, ob das Lebewesen lebt
     *
     * @return True, wenn das Lebewesen lebt
     */
    public final boolean isDead() {
        return isDead;
    }

    /**
     * Methode zum Töten von Lebewesen
     */
    private final void setDead() {
        isDead = true;
    }

    /**
     * Methode zum Abfragen der aktuellen Stufe/Level (1-3)
     *
     * @return Aktuelle Stufe/Level (0-2)
     */
    public final int getLevel()
    {
        int level = (int)(((System.nanoTime()-startTime) / 1000000000.0) / (getMaxLifetime() / 3));

        if (level > 2)
            level = 2;

        return level;
    }

    /**
     * Methode zum Rendern des Lebewesens
     * Sollte überschrieben werden
     */
    public void render() 
    {
        stroke(0);
        fill(128);
        ellipse (pos.x, pos.y, 10, 10);
    }

    /**
     * Methode zum Abfragen der Geburtszeit
     *
     * @return Geburtszeit in Nanosekunden
     */
    public final long getStartTime()
    {
        return this.startTime;
    }

    /**
     * Methode zum Aktualisieren des Lebewesens
     * Sollte überschrieben werden
     */
    public void update() 
    {
    }
}

